
// Adjust the screen lookup coordinate to make the screen look curved.
vec2 Curved(float curvature, vec2 coord)
{
    return vec2(
        coord.x - curvature * -(coord.x - 0.5) * pow((coord.y - 0.5), 2.0),
        coord.y - curvature * -(coord.y - 0.5) * pow((coord.x - 0.5), 2.0));
}

vec3 SampleCRT(float gamma, vec2 coord)
{
    return pow(abs(texture(Texture0, coord).rgb), vec3(gamma));
}

float Scanline(float y)
{
    // Increase this value to make the scanline thinner, and the black strip wider.
    const float THINNESS = 1.0;
    float resolutionY = 270.0;
    float scanlineY = mod(resolutionY * y, 1.0);
    return 1.0 - saturate(THINNESS * abs(scanlineY - 0.5));
}

void main()
{
    const float ENABLE = 1.0;
    const float GAMMA = 2.2;
    const float BLUR = 1.0;
    const float SCANLINE_SAMPLES = 2.0;
    const float CURVATURE = 0.05;
    const float CORNER_RADIUS = 0.13;
    const float EDGE_RADIUS = 0.002;
    const float BRIGHTENING = 1.4;

    if (ENABLE == 0.0)
    {
        Output = texture(Texture0, fragTexCoord0);
        return;
    }
    
    const vec2 resolution = vec2(360.0, 270.0);
    const vec2 dxy = 1.0 / resolution;
    const float dx = dxy.x;
    const float dy = dxy.y;

    // Optionally, enable curvature:
    float curvature = fragTexCoord1.x > 0.0 ? CURVATURE : 0.0;
    vec2 coord = Curved(curvature, fragTexCoord0);

    // If this pixel is outside the curved screen, produce a transparent pixel.
    if (coord.x < 0.0 || coord.x > 1.0 || coord.y < 0.0 || coord.y > 1.0)
    {
        Output = vec4(0.0);
        return;
    }

    // Distance to nearest edge:
    vec2 edge = vec2(min(coord.x, 1.0 - coord.x), 0.75 * min(coord.y, 1.0 - coord.y));
    float cornerDist = sqrt(edge.x) + sqrt(edge.y);
    float edgeDist = min(edge.x, edge.y);

    vec3 source = vec3(0.0);
    if (BLUR != 0.0)
    {
        source += 0.03 * SampleCRT(GAMMA, coord + vec2(-dx, 0.0));
        source += 1.00 * SampleCRT(GAMMA, coord + vec2(0.0, 0.0));
        source += 0.03 * SampleCRT(GAMMA, coord + vec2(dx, 0.0));
        source /= 1.06;
    }
    else
    {
        source = SampleCRT(GAMMA, coord);
    }

    float scanline = 0.0;
    if (SCANLINE_SAMPLES > 1.0)
    {
        // Calculate the scanline using multisampling; this avoid moire patterns
        // caused by the high frequency and contrast of the scanline pattern.
        for (float i = 0.0; i < SCANLINE_SAMPLES; i += 1.0)
        {
            float offset = (2.0 * i - (SCANLINE_SAMPLES - 1.0)) / (4.0 * SCANLINE_SAMPLES);
            scanline += Scanline(Curved(curvature, fragTexCoord0 + vec2(0.0, offset) * dxy).y);
        }
        scanline *= (1.0 / SCANLINE_SAMPLES);
    }
    else
    {
        scanline = Scanline(coord.y);
    }

    // Make the corners round and fade out softly at all the edges:
    float edginess = 1.0;
    edginess = smoothstep(CORNER_RADIUS, CORNER_RADIUS + 0.03, cornerDist);
    edginess *= smoothstep(0.0, EDGE_RADIUS, edgeDist);

    vec4 color;
    color.rgb = scanline * source * BRIGHTENING;
    color.a = 1.0;
    color *= edginess;
    color.rgb = pow(abs(color.rgb), vec3(1.0 / GAMMA));
    Output = color;
}
